import type { Metadata } from 'next'
import { Inter } from 'next/font/google'
import './globals.css'
import { ReactQueryProvider } from './providers';
import { Providers } from './client-providers';
import { StructuredData } from '@/components/StructuredData';
import { WhatsAppButton } from '@/components/WhatsAppButton';
import { ScrollToTop } from '@/components/ScrollToTop';
import { PageTransition } from '@/components/PageTransition';
import { Header } from '@/components/layout/Header';
import { Footer } from '@/components/layout/Footer';
import Script from 'next/script'

const inter = Inter({ subsets: ['latin'] })

export const metadata: Metadata = {
  metadataBase: new URL('https://ahericare.com'),
  title: {
    default: 'Ahericare - Professional Home Nursing & Healthcare Services in Nairobi, Kenya',
    template: '%s | Ahericare'
  },
  description: 'Leading home nursing and healthcare services in Nairobi, Kenya. Expert home care, home health services, elderly care, IV therapy, wound care, and professional nursing at home. Available 24/7 across Nairobi and Kenya.',
  keywords: [
    'home nursing Kenya',
    'home care Nairobi',
    'home health services',
    'professional nursing services',
    'elderly care Kenya',
    'home nursing Nairobi',
    'IV therapy at home',
    'wound care services',
    'catheter care',
    'home healthcare Kenya',
    'nursing services Nairobi',
    'home care services',
    'medical care at home',
    'professional nurses Kenya',
    'home health Nairobi',
    'elderly care Nairobi',
    'home nursing Westlands',
    'healthcare at home Kenya'
  ],
  authors: [{ name: 'Ahericare' }],
  creator: 'Ahericare',
  publisher: 'Ahericare',
  formatDetection: {
    email: false,
    address: false,
    telephone: false,
  },
  openGraph: {
    type: 'website',
    locale: 'en_KE',
    url: 'https://ahericare.com',
    title: 'Ahericare - Professional Home Nursing & Healthcare Services in Nairobi, Kenya',
    description: 'Leading home nursing and healthcare services in Nairobi, Kenya. Expert home care, IV therapy, wound care, and professional nursing at home.',
    siteName: 'Ahericare',
  },
  twitter: {
    card: 'summary_large_image',
    title: 'Ahericare - Professional Home Nursing & Healthcare Services',
    description: 'Expert home nursing and healthcare services in Nairobi, Kenya. Available 24/7.',
  },
  robots: {
    index: true,
    follow: true,
    googleBot: {
      index: true,
      follow: true,
      'max-video-preview': -1,
      'max-image-preview': 'large',
      'max-snippet': -1,
    },
  },
  verification: {
    google: 'google-site-verification-code',
  },
  icons: {
    icon: '/favicon.ico',
  },
}

export default function RootLayout({
  children,
}: {
  children: React.ReactNode
}) {
  return (
    <html lang="en">
      <head>
        <meta name="google-site-verification" content="googleee2d84bad5f8e307" />
        <StructuredData />
      </head>
      <body className={inter.className}>
        {/* First GA4 ID */}
        <Script
          src="https://www.googletagmanager.com/gtag/js?id=G-HKEQTHNCWX"
          strategy="afterInteractive"
        />
        <Script id="google-analytics-1" strategy="afterInteractive">
          {`
            window.dataLayer = window.dataLayer || [];
            function gtag(){dataLayer.push(arguments);}
            gtag('js', new Date());
            gtag('config', 'G-HKEQTHNCWX');
          `}
        </Script>

        {/* Second GA4 ID */}
        <Script
          src="https://www.googletagmanager.com/gtag/js?id=G-E3Q8FGE8ME"
          strategy="afterInteractive"
        />
        <Script id="google-analytics-2" strategy="afterInteractive">
          {`
            window.dataLayer = window.dataLayer || [];
            function gtag(){dataLayer.push(arguments);}
            gtag('js', new Date());
            gtag('config', 'G-E3Q8FGE8ME');
          `}
        </Script>
        <ReactQueryProvider>
          <Providers>
            <Header />
            <ScrollToTop />
            <PageTransition>
              {children}
            </PageTransition>
            <Footer />
            <WhatsAppButton />
          </Providers>
        </ReactQueryProvider>
      </body>
    </html>
  )
}
