'use client';

import { useState, useEffect, useRef } from 'react';
import { Header } from '@/components/layout/Header';
import { Footer } from '@/components/layout/Footer';
import { X } from 'lucide-react';
import Image, { type StaticImageData } from 'next/image';
import { usePathname } from 'next/navigation';
import { createPortal } from 'react-dom';
import img1 from '@/assets/gallery/DSC_-03743.jpg';
import img2 from '@/assets/gallery/DSC_-03789.jpg';
import img3 from '@/assets/gallery/DSC_-03898.jpg';
import img4 from '@/assets/gallery/DSC_-03899.jpg';
import img5 from '@/assets/gallery/DSC_-03903.jpg';
import img6 from '@/assets/gallery/DSC_-03924.jpg';
import img7 from '@/assets/gallery/DSC_-03928.jpg';
import img8 from '@/assets/gallery/DSC_-03929.jpg';
import img9 from '@/assets/gallery/DSC_-03931.jpg';
import img10 from '@/assets/gallery/DSC_-03936.jpg';
import img11 from '@/assets/gallery/DSC_-03938.jpg';
import img12 from '@/assets/gallery/DSC_-03943.jpg';
import img13 from '@/assets/gallery/DSC_-03949.jpg';
import img14 from '@/assets/gallery/DSC_-03955.jpg';
import img15 from '@/assets/gallery/DSC_-06356.jpg';
import img16 from '@/assets/gallery/DSC_-06358.jpg';
import img17 from '@/assets/gallery/DSC_-06371.jpg';
import img18 from '@/assets/gallery/DSC_-06519.jpg';
import img19 from '@/assets/gallery/DSC_-06531.jpg';

const galleryImages = [
  { src: img1, alt: 'Ahericare home healthcare service' },
  { src: img2, alt: 'Professional nursing care in Nairobi' },
  { src: img3, alt: 'Compassionate elderly care' },
  { src: img4, alt: 'Home health services' },
  { src: img5, alt: 'Medical care at home' },
  { src: img6, alt: 'Quality healthcare services' },
  { src: img7, alt: 'Patient care and comfort' },
  { src: img8, alt: 'Professional medical assistance' },
  { src: img9, alt: 'Home nursing services' },
  { src: img10, alt: 'Healthcare excellence' },
  { src: img11, alt: 'Dedicated nursing staff' },
  { src: img12, alt: 'Home health solutions' },
  { src: img13, alt: 'Compassionate care team' },
  { src: img14, alt: 'Medical services at home' },
  { src: img15, alt: 'Professional healthcare providers' },
  { src: img16, alt: 'Quality home nursing' },
  { src: img17, alt: 'Patient-centered care' },
  { src: img18, alt: 'Home healthcare excellence' },
  { src: img19, alt: 'Trusted medical services' },
];

function shuffle<T>(arr: T[]) {
  const a = arr.slice();
  for (let i = a.length - 1; i > 0; i--) {
    const j = Math.floor(Math.random() * (i + 1));
    [a[i], a[j]] = [a[j], a[i]];
  }
  return a;
}

function Lightbox({ selectedImage, onClose }: { selectedImage: StaticImageData | string | null; onClose: () => void }) {
  useEffect(() => {
    const prev = document.body.style.overflow;
    document.body.style.overflow = 'hidden';
    return () => {
      document.body.style.overflow = prev || '';
    };
  }, []);

  if (!selectedImage) return null;

  return createPortal(
    <div className="fixed inset-0 z-50 bg-foreground/90 flex items-center justify-center p-4" onClick={onClose} role="dialog" aria-modal="true">
      <button
        className="absolute top-4 right-4 p-2 bg-primary-foreground/20 rounded-full hover:bg-primary-foreground/30 transition-colors"
        onClick={onClose}
        aria-label="Close"
      >
        <X className="h-6 w-6 text-primary-foreground" />
      </button>
      <Image
        src={selectedImage as StaticImageData}
        alt="Gallery image"
        className="max-w-full max-h-[90vh] object-contain rounded-lg"
        width={1200}
        height={800}
        placeholder="blur"
        priority
        onClick={(e) => e.stopPropagation()}
      />
    </div>,
    document.body
  );
}

export default function Gallery() {
  const [selectedImage, setSelectedImage] = useState<StaticImageData | null>(null);
  // Initialize with deterministic order to avoid hydration mismatch between server and client
  const [shuffledImages, setShuffledImages] = useState(() => galleryImages);
  const pathname = usePathname();
  const gridRef = useRef<HTMLDivElement | null>(null);

  useEffect(() => {
    const grid = gridRef.current;

    // FLIP animation: measure positions, shuffle, then animate transforms
    const doShuffle = () => {
      if (!grid) {
        setShuffledImages(shuffle(galleryImages));
        return;
      }

      const nodes = Array.from(grid.children) as HTMLElement[];
      const rects = new Map<string, DOMRect>();
      nodes.forEach((node) => {
        const id = node.getAttribute('data-id');
        if (id) rects.set(id, node.getBoundingClientRect());
      });

      const newOrder = shuffle(galleryImages);
      setShuffledImages(newOrder);

      requestAnimationFrame(() => {
        const newNodes = Array.from(grid.children) as HTMLElement[];
        newNodes.forEach((node) => {
          const id = node.getAttribute('data-id');
          if (!id) return;
          const firstRect = rects.get(id);
          const newRect = node.getBoundingClientRect();
          if (firstRect) {
            const dx = firstRect.left - newRect.left;
            const dy = firstRect.top - newRect.top;
            if (dx !== 0 || dy !== 0) {
              node.style.transition = 'transform 350ms ease';
              node.style.transform = `translate(${dx}px, ${dy}px)`;
              // force layout
              node.getBoundingClientRect();
              node.style.transform = '';

              const cleanup = () => {
                node.style.transition = '';
                node.removeEventListener('transitionend', cleanup);
              };

              node.addEventListener('transitionend', cleanup);
            }
          }
        });
      });
    };

    doShuffle();
  }, [pathname]);

  return (
    <div className="min-h-screen">
      <Header />
      
      <main>
        {/* Hero Section */}
        <section className="pt-32 pb-16 bg-primary">
          <div className="container-custom">
            <div className="text-center max-w-3xl mx-auto">
              <h1 className="font-display text-4xl md:text-5xl lg:text-6xl font-bold text-primary-foreground mb-6">
                Explore Our Gallery
              </h1>
              <p className="text-primary-foreground/80 text-lg">
                Discover beautiful moments captured in time.
              </p>
            </div>
          </div>
        </section>

        {/* Gallery Grid */}
        <section className="section-padding bg-background">
          <div className="container-custom">
            <div ref={gridRef} className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
              {shuffledImages.map((image, index) => (
                <button
                  data-id={typeof image.src === 'string' ? image.src : image.src.src}
                  key={typeof image.src === 'string' ? image.src : image.src.src}
                  onClick={() => setSelectedImage(image.src)}
                  className="relative aspect-square overflow-hidden rounded-xl group cursor-pointer"
                >
                  <Image
                    src={image.src}
                    alt={image.alt}
                    className="w-full h-full object-cover transition-transform duration-500 group-hover:scale-110"
                    width={500}
                    height={500}
                    sizes="(min-width: 1024px) 25vw, (min-width: 768px) 33vw, 50vw"
                    placeholder="blur"
                    loading={index < 8 ? 'eager' : 'lazy'}
                    priority={index < 4}
                    quality={75}
                  />
                  <div className="absolute inset-0 bg-primary/0 group-hover:bg-primary/40 transition-colors duration-300 flex items-center justify-center">
                    <span className="text-primary-foreground font-semibold opacity-0 group-hover:opacity-100 transition-opacity duration-300 text-center px-4">
                      {image.alt}
                    </span>
                  </div>
                </button>
              ))} 
            </div>
          </div>
        </section>

        {/* Lightbox */}
        <Lightbox selectedImage={selectedImage} onClose={() => setSelectedImage(null)} />
      </main>
    </div>
  );
}
