'use client';

import { useEffect, useState } from 'react';
import { useRouter } from 'next/navigation';
import Link from 'next/link';
import Image from 'next/image';
import { supabase, GalleryImage } from '@/lib/supabase';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Switch } from '@/components/ui/switch';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from '@/components/ui/dialog';
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from '@/components/ui/alert-dialog';
import { toast } from 'sonner';
import { 
  Loader2, 
  Plus, 
  ArrowLeft,
  Image as ImageIcon,
  Edit,
  Trash2,
  Eye,
  EyeOff,
  Save
} from 'lucide-react';

export default function GalleryManagementPage() {
  const router = useRouter();
  const [loading, setLoading] = useState(true);
  const [images, setImages] = useState<GalleryImage[]>([]);
  const [saving, setSaving] = useState(false);
  const [editingImage, setEditingImage] = useState<GalleryImage | null>(null);
  const [newImage, setNewImage] = useState({
    title: '',
    description: '',
    image_url: '',
    display_order: 0,
    visible: true,
  });
  const [addDialogOpen, setAddDialogOpen] = useState(false);
  const [editDialogOpen, setEditDialogOpen] = useState(false);

  useEffect(() => {
    checkAuthAndFetch();
  }, []);

  async function checkAuthAndFetch() {
    const { data: { session } } = await supabase.auth.getSession();
    
    if (!session) {
      router.push('/admin');
      return;
    }

    const { data: adminUser } = await supabase
      .from('admin_users')
      .select('*')
      .eq('user_id', session.user.id)
      .single();

    if (!adminUser) {
      await supabase.auth.signOut();
      router.push('/admin');
      return;
    }

    await fetchImages();
  }

  async function fetchImages() {
    const { data, error } = await supabase
      .from('gallery_images')
      .select('*')
      .order('display_order', { ascending: true });

    if (!error && data) {
      setImages(data as GalleryImage[]);
    }
    setLoading(false);
  }

  async function handleAddImage(e: React.FormEvent) {
    e.preventDefault();
    setSaving(true);

    try {
      const { error } = await supabase
        .from('gallery_images')
        .insert(newImage);

      if (error) throw error;

      toast.success('Image added successfully!');
      setAddDialogOpen(false);
      setNewImage({
        title: '',
        description: '',
        image_url: '',
        display_order: 0,
        visible: true,
      });
      await fetchImages();
    } catch (error: any) {
      toast.error(error.message || 'Failed to add image');
    }
    setSaving(false);
  }

  async function handleUpdateImage(e: React.FormEvent) {
    e.preventDefault();
    if (!editingImage) return;
    setSaving(true);

    try {
      const { error } = await supabase
        .from('gallery_images')
        .update({
          title: editingImage.title,
          description: editingImage.description,
          image_url: editingImage.image_url,
          display_order: editingImage.display_order,
          visible: editingImage.visible,
        })
        .eq('id', editingImage.id);

      if (error) throw error;

      toast.success('Image updated successfully!');
      setEditDialogOpen(false);
      setEditingImage(null);
      await fetchImages();
    } catch (error: any) {
      toast.error(error.message || 'Failed to update image');
    }
    setSaving(false);
  }

  async function handleDeleteImage(id: string) {
    const { error } = await supabase
      .from('gallery_images')
      .delete()
      .eq('id', id);

    if (error) {
      toast.error('Failed to delete image');
    } else {
      toast.success('Image deleted successfully');
      setImages(images.filter(img => img.id !== id));
    }
  }

  async function toggleVisibility(image: GalleryImage) {
    const { error } = await supabase
      .from('gallery_images')
      .update({ visible: !image.visible })
      .eq('id', image.id);

    if (error) {
      toast.error('Failed to update visibility');
    } else {
      setImages(images.map(img => 
        img.id === image.id ? { ...img, visible: !img.visible } : img
      ));
    }
  }

  if (loading) {
    return (
      <main className="min-h-screen bg-background flex items-center justify-center pt-24">
        <Loader2 className="h-8 w-8 animate-spin text-primary" />
      </main>
    );
  }

  return (
    <main className="min-h-screen bg-muted/30">
      {/* Hero Background */}
      <div className="bg-[#7D2C5A] h-32" />
      
      <div className="container mx-auto px-4 py-8">
        {/* Header */}
        <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4 mb-6">
          <div>
            <Link 
              href="/admin/dashboard" 
              className="inline-flex items-center text-muted-foreground hover:text-foreground mb-2"
            >
              <ArrowLeft className="h-4 w-4 mr-2" />
              Back to Dashboard
            </Link>
            <h1 className="text-3xl font-bold text-foreground flex items-center gap-2">
              <ImageIcon className="h-8 w-8 text-primary" />
              Gallery Management
            </h1>
          </div>
          
          <Dialog open={addDialogOpen} onOpenChange={setAddDialogOpen}>
            <DialogTrigger asChild>
              <Button>
                <Plus className="h-4 w-4 mr-2" />
                Add Image
              </Button>
            </DialogTrigger>
            <DialogContent>
              <DialogHeader>
                <DialogTitle>Add New Image</DialogTitle>
              </DialogHeader>
              <form onSubmit={handleAddImage} className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="new_image_url">Image URL *</Label>
                  <Input
                    id="new_image_url"
                    placeholder="https://example.com/image.jpg"
                    value={newImage.image_url}
                    onChange={(e) => setNewImage({ ...newImage, image_url: e.target.value })}
                    required
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="new_title">Title</Label>
                  <Input
                    id="new_title"
                    placeholder="Image title"
                    value={newImage.title}
                    onChange={(e) => setNewImage({ ...newImage, title: e.target.value })}
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="new_description">Description</Label>
                  <Textarea
                    id="new_description"
                    placeholder="Image description"
                    value={newImage.description}
                    onChange={(e) => setNewImage({ ...newImage, description: e.target.value })}
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="new_order">Display Order</Label>
                  <Input
                    id="new_order"
                    type="number"
                    value={newImage.display_order}
                    onChange={(e) => setNewImage({ ...newImage, display_order: parseInt(e.target.value) || 0 })}
                  />
                </div>
                <div className="flex items-center justify-between">
                  <Label htmlFor="new_visible">Visible</Label>
                  <Switch
                    id="new_visible"
                    checked={newImage.visible}
                    onCheckedChange={(checked) => setNewImage({ ...newImage, visible: checked })}
                  />
                </div>
                <Button type="submit" className="w-full" disabled={saving}>
                  {saving ? <Loader2 className="h-4 w-4 animate-spin" /> : 'Add Image'}
                </Button>
              </form>
            </DialogContent>
          </Dialog>
        </div>

        {/* Gallery Grid */}
        {images.length === 0 ? (
          <Card>
            <CardContent className="text-center py-12">
              <ImageIcon className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
              <p className="text-muted-foreground">No images in gallery</p>
              <Button className="mt-4" onClick={() => setAddDialogOpen(true)}>
                <Plus className="h-4 w-4 mr-2" />
                Add your first image
              </Button>
            </CardContent>
          </Card>
        ) : (
          <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
            {images.map((image) => (
              <Card key={image.id} className={`overflow-hidden ${!image.visible ? 'opacity-60' : ''}`}>
                <div className="relative aspect-square">
                  <Image
                    src={image.image_url}
                    alt={image.title || 'Gallery image'}
                    fill
                    className="object-cover"
                  />
                  {!image.visible && (
                    <div className="absolute inset-0 bg-background/50 flex items-center justify-center">
                      <EyeOff className="h-8 w-8 text-muted-foreground" />
                    </div>
                  )}
                </div>
                <CardContent className="p-3">
                  <p className="font-medium text-sm line-clamp-1">{image.title || 'Untitled'}</p>
                  <p className="text-xs text-muted-foreground">Order: {image.display_order}</p>
                  <div className="flex gap-1 mt-2">
                    <Button 
                      variant="ghost" 
                      size="icon" 
                      className="h-8 w-8"
                      onClick={() => toggleVisibility(image)}
                    >
                      {image.visible ? <Eye className="h-4 w-4" /> : <EyeOff className="h-4 w-4" />}
                    </Button>
                    <Button 
                      variant="ghost" 
                      size="icon" 
                      className="h-8 w-8"
                      onClick={() => {
                        setEditingImage(image);
                        setEditDialogOpen(true);
                      }}
                    >
                      <Edit className="h-4 w-4" />
                    </Button>
                    <AlertDialog>
                      <AlertDialogTrigger asChild>
                        <Button variant="ghost" size="icon" className="h-8 w-8 text-destructive">
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </AlertDialogTrigger>
                      <AlertDialogContent>
                        <AlertDialogHeader>
                          <AlertDialogTitle>Delete Image?</AlertDialogTitle>
                          <AlertDialogDescription>
                            This action cannot be undone.
                          </AlertDialogDescription>
                        </AlertDialogHeader>
                        <AlertDialogFooter>
                          <AlertDialogCancel>Cancel</AlertDialogCancel>
                          <AlertDialogAction
                            onClick={() => handleDeleteImage(image.id)}
                            className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
                          >
                            Delete
                          </AlertDialogAction>
                        </AlertDialogFooter>
                      </AlertDialogContent>
                    </AlertDialog>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        )}

        {/* Edit Dialog */}
        <Dialog open={editDialogOpen} onOpenChange={setEditDialogOpen}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Edit Image</DialogTitle>
            </DialogHeader>
            {editingImage && (
              <form onSubmit={handleUpdateImage} className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="edit_image_url">Image URL *</Label>
                  <Input
                    id="edit_image_url"
                    value={editingImage.image_url}
                    onChange={(e) => setEditingImage({ ...editingImage, image_url: e.target.value })}
                    required
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="edit_title">Title</Label>
                  <Input
                    id="edit_title"
                    value={editingImage.title || ''}
                    onChange={(e) => setEditingImage({ ...editingImage, title: e.target.value })}
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="edit_description">Description</Label>
                  <Textarea
                    id="edit_description"
                    value={editingImage.description || ''}
                    onChange={(e) => setEditingImage({ ...editingImage, description: e.target.value })}
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="edit_order">Display Order</Label>
                  <Input
                    id="edit_order"
                    type="number"
                    value={editingImage.display_order}
                    onChange={(e) => setEditingImage({ ...editingImage, display_order: parseInt(e.target.value) || 0 })}
                  />
                </div>
                <div className="flex items-center justify-between">
                  <Label htmlFor="edit_visible">Visible</Label>
                  <Switch
                    id="edit_visible"
                    checked={editingImage.visible}
                    onCheckedChange={(checked) => setEditingImage({ ...editingImage, visible: checked })}
                  />
                </div>
                <Button type="submit" className="w-full" disabled={saving}>
                  {saving ? <Loader2 className="h-4 w-4 animate-spin" /> : (
                    <>
                      <Save className="h-4 w-4 mr-2" />
                      Save Changes
                    </>
                  )}
                </Button>
              </form>
            )}
          </DialogContent>
        </Dialog>
      </div>
    </main>
  );
}
