'use client';

import { useState } from 'react';
import { useRouter } from 'next/navigation';
import Link from 'next/link';
import { supabase } from '@/lib/supabase';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Switch } from '@/components/ui/switch';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { ImageUpload } from '@/components/admin/ImageUpload';
import { toast } from 'sonner';
import { ArrowLeft, Loader2, Save } from 'lucide-react';

export default function NewBlogPage() {
  const router = useRouter();
  const [saving, setSaving] = useState(false);
  const [form, setForm] = useState({
    title: '',
    slug: '',
    excerpt: '',
    content: '',
    cover_image: '',
    author_name: 'Ahericare Team',
    author_avatar: '',
    published: false,
  });

  const generateSlug = (title: string) => {
    return title
      .toLowerCase()
      .replace(/[^a-z0-9]+/g, '-')
      .replace(/(^-|-$)/g, '');
  };

  const handleTitleChange = (title: string) => {
    setForm({
      ...form,
      title,
      slug: generateSlug(title),
    });
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setSaving(true);

    try {
      const { error } = await supabase
        .from('blogs')
        .insert({
          ...form,
          published_at: form.published ? new Date().toISOString() : null,
        });

      if (error) throw error;

      toast.success('Blog created successfully!');
      router.push('/admin/blogs');
    } catch (error: any) {
      toast.error(error.message || 'Failed to create blog');
      setSaving(false);
    }
  };

  return (
    <main className="min-h-screen bg-muted/30">
      {/* Hero Background */}
      <div className="bg-[#7D2C5A] h-32" />
      
      <div className="container mx-auto px-4 max-w-4xl py-8">
        {/* Header */}
        <div className="mb-6">
          <Link 
            href="/admin/blogs" 
            className="inline-flex items-center text-muted-foreground hover:text-foreground mb-2"
          >
            <ArrowLeft className="h-4 w-4 mr-2" />
            Back to Blogs
          </Link>
          <h1 className="text-3xl font-bold text-foreground">Create New Blog</h1>
        </div>

        <form onSubmit={handleSubmit}>
          <div className="grid gap-6">
            {/* Main Content */}
            <Card>
              <CardHeader>
                <CardTitle>Blog Content</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="title">Title *</Label>
                  <Input
                    id="title"
                    placeholder="Enter blog title"
                    value={form.title}
                    onChange={(e) => handleTitleChange(e.target.value)}
                    required
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="slug">Slug *</Label>
                  <Input
                    id="slug"
                    placeholder="blog-post-url"
                    value={form.slug}
                    onChange={(e) => setForm({ ...form, slug: e.target.value })}
                    required
                  />
                  <p className="text-sm text-muted-foreground">
                    URL: /blog/{form.slug || 'your-slug'}
                  </p>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="excerpt">Excerpt</Label>
                  <Textarea
                    id="excerpt"
                    placeholder="Brief summary of the blog post..."
                    value={form.excerpt}
                    onChange={(e) => setForm({ ...form, excerpt: e.target.value })}
                    rows={3}
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="content">Content *</Label>
                  <Textarea
                    id="content"
                    placeholder="Write your blog content here...

Use markdown-like formatting:
# Heading 1
## Heading 2
### Heading 3

Regular paragraphs are separated by blank lines.

- List item 1
- List item 2"
                    value={form.content}
                    onChange={(e) => setForm({ ...form, content: e.target.value })}
                    rows={15}
                    required
                    className="font-mono text-sm"
                  />
                </div>

                <ImageUpload
                  label="Cover Image"
                  value={form.cover_image}
                  onChange={(url) => setForm({ ...form, cover_image: url })}
                  folder="covers"
                  placeholder="https://example.com/image.jpg"
                />
              </CardContent>
            </Card>

            {/* Author & Settings */}
            <Card>
              <CardHeader>
                <CardTitle>Author & Settings</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid md:grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label htmlFor="author_name">Author Name</Label>
                    <Input
                      id="author_name"
                      placeholder="Author Name"
                      value={form.author_name}
                      onChange={(e) => setForm({ ...form, author_name: e.target.value })}
                    />
                  </div>
                  <ImageUpload
                    label="Author Avatar"
                    value={form.author_avatar}
                    onChange={(url) => setForm({ ...form, author_avatar: url })}
                    folder="avatars"
                    placeholder="https://example.com/avatar.jpg"
                  />
                </div>

                <div className="flex items-center justify-between p-4 bg-muted rounded-lg">
                  <div>
                    <Label htmlFor="published" className="font-medium">Publish immediately</Label>
                    <p className="text-sm text-muted-foreground">
                      Make this blog visible to the public
                    </p>
                  </div>
                  <Switch
                    id="published"
                    checked={form.published}
                    onCheckedChange={(checked) => setForm({ ...form, published: checked })}
                  />
                </div>
              </CardContent>
            </Card>

            {/* Actions */}
            <div className="flex gap-3 justify-end">
              <Link href="/admin/blogs">
                <Button type="button" variant="outline">Cancel</Button>
              </Link>
              <Button type="submit" disabled={saving}>
                {saving ? (
                  <>
                    <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                    Saving...
                  </>
                ) : (
                  <>
                    <Save className="h-4 w-4 mr-2" />
                    Create Blog
                  </>
                )}
              </Button>
            </div>
          </div>
        </form>
      </div>
    </main>
  );
}
